/*
Classic Cinema Animation (CCA) (c) Beyond The Hype Youtube Channel

This work is licensed under the Creative Commons
Attribution-ShareAlike 4.0 International License.
To view a copy of this license, visit
http://creativecommons.org/licenses/by-sa/4.0/
*/

// version 1.0.0

	 /// MENU ///

#include "ReShadeUI.fxh"

uniform int3 Size < __UNIFORM_INPUT_INT3
	ui_label = "X frames, Y frames, FPS";
	ui_tooltip = "Adjust CCA texture dimensions and framerate\n"
		"To change texture resolution and name,\n"
		"add following preprocessor definition:\n"
		"  CCA_TEXTURE 'name.png'\n"
		"  CCA_SIZE_X [ResolutionX]\n"
		"  CCA_SIZE_Y [ResolutionY]";
	#if __RESHADE__ < 40000
		ui_step = 0.2;
	#endif
	ui_min = 1; ui_max = 30;
	ui_category = "Texture dimensions";
> = int3(9, 9, 12);

uniform float3 Position < __UNIFORM_SLIDER_FLOAT3
	ui_label = "X position, Y position, Scale";
	ui_tooltip = "Adjust CCA texture size and position";
	#if __RESHADE__ < 40000
		ui_step = 0.002;
	#endif
	ui_min = float3(0.0, 0.0, 0.1); ui_max = float3(1.0, 1.0, 1.0);
	ui_category = "Position on screen";
> = float3(0.0, 0.0, 1.0);

// Get time in milliseconds from start
uniform float timer < source = "timer"; >;

//-------------------------------------------------------------------------------------------------

#ifndef CCA_PIXELATED
	#define CCA_PIXELATED 0
#endif

#ifndef CCA_TEXTURE
	#define CCA_TEXTURE "bthcustomgrey20.png" // Texture file name
#endif

#ifndef Classic_Cinema_Animation_SIZE_X
	#define Classic_Cinema_Animation_SIZE_X 3840 // Texture horizontal resolution
#endif

#ifndef Classic_Cinema_Animation_SIZE_Y
	#define Classic_Cinema_Animation_SIZE_Y 2160 // Texture vertical resolution
#endif

//-------------------------------------------------------------------------------------------------


	 /// SHADER ///

texture ccaTex < source = CCA_TEXTURE; > {Width = Classic_Cinema_Animation_SIZE_X; Height = Classic_Cinema_Animation_SIZE_Y;};
sampler ccaSampler
{
	Texture = ccaTex;
	AddressU = REPEAT;
	AddressV = REPEAT;
	#if CCA_PIXELATED==1
		MagFilter = POINT;
		MinFilter = POINT;
		MipFilter = POINT;
	#endif
};

#include "ReShade.fxh"

float Mask(float2 Coord)
{
	Coord = abs(Coord*2.0-1.0);
	float2 Pixel = fwidth(Coord);
	float2 Borders = 1.0 - smoothstep(1.0-Pixel, 1.0+Pixel, Coord);
	return min(Borders.x, Borders.y);
}

float3 ccaPS(float4 vois : SV_Position, float2 texcoord : TexCoord) : SV_Target
{
	float ScreenAspect = ReShade::AspectRatio;
	// Screen aspect divided by animation frame aspect
	float AspectDifference = (ScreenAspect*float(Size.x*Classic_Cinema_Animation_SIZE_Y))/float(Size.y*Classic_Cinema_Animation_SIZE_X);

	// Scale coordinates
	float2 Scale = 1.0/Position.z;
	float2 ScaledCoord = texcoord * Scale;

	// Adjust aspect ratio
	if(AspectDifference > 1.0)
	{
		ScaledCoord.x *= AspectDifference;
		Scale.x *= AspectDifference;
	}
	else if(AspectDifference < 1.0)
	{
		ScaledCoord.y /= AspectDifference;
		Scale.y /= AspectDifference;
	}

	// Offset coordinates
	ScaledCoord += (1.0-Scale)*float2(Position.x, 1.0-Position.y);

	float BorderMask = Mask(ScaledCoord);
	// Frame time in milliseconds
	float FramerateInMs = 1000.0 / Size.z;
	float2 AnimationCoord = ScaledCoord / Size.xy;
	// Sample UVs for horizontal and vertical frames
	AnimationCoord.x += frac(floor(timer / FramerateInMs) / Size.x);
	AnimationCoord.y += frac(floor( timer / (FramerateInMs * Size.x) )/Size.y);

	// Sample display image
	float3 Display = tex2D(ReShade::BackBuffer, texcoord).rgb;
	// Sample Classic Cinema texture
	float4 AnimationTexture = tex2D(ccaSampler, AnimationCoord);

	return lerp(Display, AnimationTexture.rgb, AnimationTexture.a * BorderMask);
}


	 /// OUTPUT ///

technique CCA < ui_tooltip = "Classic Cinema Animation FX:\n"
	"------------------\n"
	"To change texture resolution and name,\n"
	"change the following preprocessor definitions:\n"
	"  CCA 'name.png'\n"
	"  CCA_SIZE_X [ResolutionX]\n"
	"  CCA_SIZE_Y [ResolutionY]"; >
{
	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = ccaPS;
	}
}
